/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.util.io;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.CopyOption;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Stack;
import java.util.Vector;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FilenameUtils;
import org.jabref.logic.bibtexkeypattern.BracketedPattern;
import org.jabref.logic.util.io.FileNameCleaner;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.util.OptionalUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtil {
    public static final boolean IS_POSIX_COMPILANT = FileSystems.getDefault().supportedFileAttributeViews().contains("posix");
    public static final int MAXIMUM_FILE_NAME_LENGTH = 255;
    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtil.class);

    private FileUtil() {
    }

    public static Optional<String> getFileExtension(String fileName) {
        int dotPosition = fileName.lastIndexOf(46);
        if (dotPosition > 0 && dotPosition < fileName.length() - 1) {
            return Optional.of(fileName.substring(dotPosition + 1).trim().toLowerCase(Locale.ROOT));
        }
        return Optional.empty();
    }

    public static Optional<String> getFileExtension(File file) {
        return FileUtil.getFileExtension(file.getName());
    }

    public static String getBaseName(String fileNameWithExtension) {
        return FilenameUtils.getBaseName(fileNameWithExtension);
    }

    public static String getValidFileName(String fileName) {
        String nameWithoutExtension = FileUtil.getBaseName(fileName);
        if (nameWithoutExtension.length() > 255) {
            Optional<String> extension = FileUtil.getFileExtension(fileName);
            String shortName = nameWithoutExtension.substring(0, 255);
            LOGGER.info(String.format("Truncated the too long filename '%s' (%d characters) to '%s'.", fileName, fileName.length(), shortName));
            return extension.map(s2 -> shortName + "." + s2).orElse(shortName);
        }
        return fileName;
    }

    public static Path addExtension(Path path, String extension) {
        Path fileName = path.getFileName();
        return path.resolveSibling(fileName + extension);
    }

    public static List<String> uniquePathSubstrings(List<String> paths) {
        ArrayList stackList = new ArrayList(paths.size());
        for (String path : paths) {
            List<String> directories = Arrays.asList(path.split(Pattern.quote(File.separator)));
            Stack<String> stack = new Stack<String>();
            stack.addAll(directories);
            stackList.add(stack);
        }
        ArrayList<String> pathSubstrings = new ArrayList<String>(Collections.nCopies(paths.size(), ""));
        while (!stackList.stream().allMatch(Vector::isEmpty)) {
            String tempString;
            int i;
            for (i = 0; i < stackList.size(); ++i) {
                tempString = (String)pathSubstrings.get(i);
                if (tempString.isEmpty() && !((Stack)stackList.get(i)).isEmpty()) {
                    pathSubstrings.set(i, (String)((Stack)stackList.get(i)).pop());
                    continue;
                }
                if (((Stack)stackList.get(i)).isEmpty()) continue;
                pathSubstrings.set(i, (String)((Stack)stackList.get(i)).pop() + File.separator + tempString);
            }
            for (i = 0; i < stackList.size(); ++i) {
                tempString = (String)pathSubstrings.get(i);
                if (Collections.frequency(pathSubstrings, tempString) != 1) continue;
                ((Stack)stackList.get(i)).clear();
            }
        }
        return pathSubstrings;
    }

    public static boolean copyFile(Path pathToSourceFile, Path pathToDestinationFile, boolean replaceExisting) {
        if (!Files.exists(pathToSourceFile, new LinkOption[0])) {
            LOGGER.error("Path to the source file doesn't exist.");
            return false;
        }
        if (Files.exists(pathToDestinationFile, new LinkOption[0]) && !replaceExisting) {
            LOGGER.error("Path to the destination file is not exists and the file shouldn't be replace.");
            return false;
        }
        try {
            Files.write(pathToDestinationFile, Files.readAllBytes(pathToSourceFile), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);
            return true;
        }
        catch (IOException e) {
            LOGGER.error("Copying Files failed.", e);
            return false;
        }
    }

    public static boolean renameFile(Path fromFile, Path toFile) {
        return FileUtil.renameFile(fromFile, toFile, false);
    }

    public static boolean renameFile(Path fromFile, Path toFile, boolean replaceExisting) {
        try {
            return FileUtil.renameFileWithException(fromFile, toFile, replaceExisting);
        }
        catch (IOException e) {
            LOGGER.error("Renaming Files failed", e);
            return false;
        }
    }

    public static boolean renameFileWithException(Path fromFile, Path toFile, boolean replaceExisting) throws IOException {
        if (replaceExisting) {
            return Files.move(fromFile, fromFile.resolveSibling(toFile), StandardCopyOption.REPLACE_EXISTING) != null;
        }
        return Files.move(fromFile, fromFile.resolveSibling(toFile), new CopyOption[0]) != null;
    }

    public static Path shortenFileName(Path file, List<Path> dirs) {
        if (!file.isAbsolute()) {
            return file;
        }
        for (Path dir : dirs) {
            if (!file.startsWith(dir)) continue;
            return dir.relativize(file);
        }
        return file;
    }

    public static List<Path> getListOfLinkedFiles(List<BibEntry> bes, List<Path> fileDirs) {
        Objects.requireNonNull(bes);
        Objects.requireNonNull(fileDirs);
        return bes.stream().flatMap(entry -> entry.getFiles().stream()).flatMap(file -> OptionalUtil.toStream(file.findIn(fileDirs))).collect(Collectors.toList());
    }

    public static String createFileNameFromPattern(BibDatabase database, BibEntry entry, String fileNamePattern) {
        String targetName = BracketedPattern.expandBrackets(fileNamePattern, Character.valueOf(';'), entry, database);
        if (targetName.isEmpty()) {
            targetName = entry.getCiteKeyOptional().orElse("default");
        }
        targetName = FileNameCleaner.cleanFileName(targetName);
        return targetName;
    }

    public static String createDirNameFromPattern(BibDatabase database, BibEntry entry, String fileNamePattern) {
        String targetName = BracketedPattern.expandBrackets(fileNamePattern, Character.valueOf(';'), entry, database);
        if (targetName.isEmpty()) {
            targetName = entry.getCiteKeyOptional().orElse("default");
        }
        targetName = FileNameCleaner.cleanDirectoryName(targetName);
        return targetName;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Optional<Path> find(String filename, Path rootDirectory) {
        try (Stream<Path> pathStream = Files.walk(rootDirectory, new FileVisitOption[0]);){
            Optional<Path> optional = pathStream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(f -> f.getFileName().toString().equals(filename)).findFirst();
            return optional;
        }
        catch (IOException | UncheckedIOException ex) {
            LOGGER.error("Error trying to locate the file " + filename + " inside the directory " + rootDirectory);
            return Optional.empty();
        }
    }

    public static List<Path> find(String filename, List<Path> directories) {
        ArrayList<Path> files = new ArrayList<Path>();
        for (Path dir : directories) {
            FileUtil.find(filename, dir).ifPresent(files::add);
        }
        return files;
    }

    public static String toPortableString(Path path) {
        return path.toString().replace('\\', '/');
    }
}

